/*
 * linux/arch/arm/mach-dmw/gpio.c
 *
 *  Copyright (C) 2012, DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/module.h>
#include <mach/hardware.h>
#include <mach/irqs.h>

#include "gpio-setup.h"

#define GET_REG(reg, gpio) ( \
	__raw_readl(dmw_gpio_base + (__BANK(gpio) * 0x60) + (reg)) & (1 << __PIN(gpio)) \
	)

#define SET_REG(reg, gpio) \
	__raw_writel(1 << __PIN(gpio), dmw_gpio_base + (__BANK(gpio) * 0x60) + (reg))

void *dmw_gpio_base;
static __initdata unsigned long touched_gpios[7];


static void gpio_set_enable(unsigned gpio, int value)
{
	if (value)
		SET_REG(XGP_EN_SET, gpio);
	else
		SET_REG(XGP_EN_CLR, gpio);
}

static void gpio_set_pull_enable(unsigned gpio, int value)
{
	if (value)
		SET_REG(XGP_PULL_EN, gpio);
	else
		SET_REG(XGP_PULL_DIS, gpio);
}

static void gpio_set_pull_selection(unsigned gpio, int value)
{
	if (value)
		SET_REG(XGP_PULL_UP, gpio);
	else
		SET_REG(XGP_PULL_DOWN, gpio);
}

int gpio_to_irq(unsigned gpio)
{
	if (gpio < GPIO_PORTF(16) || gpio > GPIO_PORTF(31))
		return -EINVAL;
	else if (gpio >= GPIO_PORTF(23))
		return DMW_IID_EXTERNAL_REQUEST_7 + gpio - GPIO_PORTF(23);
	else
		return DMW_IID_EXTERNAL_REQUEST_0 + gpio - GPIO_PORTF(16);
}
EXPORT_SYMBOL(gpio_to_irq);

int irq_to_gpio(unsigned int irq)
{
	if (irq >= DMW_IID_EXTERNAL_REQUEST_0 &&
	    irq <= DMW_IID_EXTERNAL_REQUEST_6)
		return GPIO_PORTF(16 + irq - DMW_IID_EXTERNAL_REQUEST_0);
	else if (irq >= DMW_IID_EXTERNAL_REQUEST_7 &&
		 irq <= DMW_IID_EXTERNAL_REQUEST_15)
		return GPIO_PORTF(23 + irq - DMW_IID_EXTERNAL_REQUEST_7);
	else
		return -EINVAL;
}
EXPORT_SYMBOL(irq_to_gpio);

/*
 * Bus recovery sequence for I2C master. The function will temporarily take the
 * ownership of the pins as GPIOs and will bit-bang the following sequence:
 *
 *   - Configure SDA as input and SCL as output
 *   - Generate up to 9 clock pulses on SCL line
 *   - After each clock pulse, check if Slave-Transmitter releases the SDA line
 *     to perform the ACK operation (SDA high)
 *   - Keeping SDA High during the ACK means that the Master-Receiver does not
 *     acknowledge the previous byte receive
 *   - The Slave-Transmitter then goes in an idle state
 *   - Sends a STOP command initializing completely the bus
 */
void dmw_i2c_recover(unsigned scl, unsigned sda)
{
	struct gpio i2c_gpios[] = {
		{ scl, GPIOF_OUT_INIT_LOW, "dmw_i2c_recover-scl" },
		{ sda, GPIOF_IN, "dmw_i2c_recover-sda" },
	};

	int i;

	gpio_set_enable(sda, 1);
	gpio_set_enable(scl, 1);

	if (gpio_request_array(i2c_gpios, ARRAY_SIZE(i2c_gpios)))
		return;

	for (i = 0; i < 10; i++) {
		gpio_set_value(scl, 0);
		udelay(5);
		gpio_set_value(scl, 1);
		udelay(5);
		if (gpio_get_value(sda) == 1)
			break;
	}

	if (gpio_get_value(sda) != 1)
		printk(KERN_WARNING "%s(): slave is still holding sda low\n",
			__func__);

	/* generate stop */
	gpio_direction_output(sda, 0);
	udelay(5);
	gpio_direction_output(sda, 1);

	udelay(100);

	gpio_direction_input(sda);
	gpio_direction_input(scl);

	gpio_free_array(i2c_gpios, ARRAY_SIZE(i2c_gpios));
	gpio_set_enable(sda, 0);
	gpio_set_enable(scl, 0);
}
EXPORT_SYMBOL(dmw_i2c_recover);

__init void dmw_gpio_setup(struct gpio_setup_t *setup, unsigned int size)
{
	for ( ; size-- ; setup++ ) {
		unsigned gpio = setup->gpio;

		if (gpio >= 7*32) {
			printk(KERN_ERR "dmw_gpio_setup: invalid GPIO: %cGPIO%d\n",
				'A' + __BANK(gpio), __PIN(gpio));
			continue;
		}

		if (test_and_set_bit(gpio, touched_gpios)) {
			printk(KERN_WARNING "dmw_gpio_setup: %cGPIO%d already "
				"configured!\n", 'A' + __BANK(gpio),
				__PIN(gpio));
			continue;
		}

		/* first enable pull-up/-down if requested */
		if (setup->pull != GPIO_PULL_DISABLE) {
			gpio_set_pull_selection(gpio, setup->pull);
			gpio_set_pull_enable(gpio, 1);
		}

		/* enable/disable GPIO */
		if (setup->state != GPIO_DISABLE) {
			int ret;

			gpio_set_enable(gpio, 1);

			ret = gpio_request(gpio, "gpio-setup");
			if (ret) {
				printk(KERN_WARNING "dmw_gpio_setup: cannot get "
					"%cGPIO%d: %d\n", 'A' + __BANK(gpio),
					__PIN(gpio), ret);
				continue;
			}

			if (setup->state == GPIO_IN)
				gpio_direction_input(gpio);
			else {
				gpio_direction_output(gpio, setup->state == GPIO_OUT_HIGH);
				if (setup->pull != GPIO_PULL_DISABLE)
					printk(KERN_WARNING "dmw_gpio_setup: %cGPIO%d: "
						"pull-%s and output enabled simultaneously!\n",
						'A' + __BANK(gpio), __PIN(gpio),
						setup->pull ? "up" : "down");
			}
			gpio_free(gpio);
		} else {
			gpio_set_enable(gpio, 0);
		}

		/* optionally disable pull-up/-down as last step */
		if (setup->pull == GPIO_PULL_DISABLE)
			gpio_set_pull_enable(gpio, 0);
	}
}

static int __init gpio_init(void)
{
	dmw_gpio_base = ioremap_nocache(DMW_GPIO_BASE, SZ_4K);
	if (!dmw_gpio_base) {
		printk(KERN_ERR "%s: Can't remap the GPIO memory space\n",
				__FUNCTION__);
		return -ENOMEM;
	}

	return 0;
}
arch_initcall(gpio_init);

static int __init dmw_gpio_setup_late(void)
{
	int gpio, ret;

	/*
	 * Set all unconfigured GPIO's as input, pull-down.
	 */
	for (gpio = 0; gpio < 7*32; gpio++) {
		if (test_bit(gpio, touched_gpios))
			continue;

		ret = gpio_request(gpio, "disabled-gpio");
		if (ret) {
			if (ret == -ENODEV)
				continue;

			printk(KERN_WARNING "dmw_gpio_setup_late: cannot invalidate "
				"%cGPIO%d!\n", 'A' + __BANK(gpio), __PIN(gpio));
			continue;
		}

		gpio_set_pull_selection(gpio, 0);
		gpio_set_pull_enable(gpio, 1);
		gpio_direction_input(gpio);
	}

	return 0;
}
late_initcall(dmw_gpio_setup_late);
