/**\file
 * This file encapsulates all the APB dependent initialization and resource allocation
 * on Linux
 */  
  
//#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/init.h>
#include <linux/dma-mapping.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/clk.h>
  
#include "synopGMAC_plat.h"
#include "synopGMAC_apb_bus_interface.h"
#include "synopGMAC_Host.h"
  
/**********************************************************************************/ 
  
#define SYNOPGMAC_VENDOR_ID  0x0700
#define SYNOPGMAC_DEVICE_ID  0x1108
  
static u8 synopGMAC_driver_name[] = "dmw96_gmac";
u32 synop_apb_using_dac;
EXPORT_SYMBOL(synop_apb_using_dac);
/**********************************************************************************/ 
extern synopGMACNetworkAdapter *synopGMACadapter;
extern u8 *synopGMACMappedAddr;
extern u32 synopGMACMappedAddrSize;
extern struct platform_device *synopGMACapbdev;


/**
 * probe function of Linux apb driver.
 * 	- Ioremap the BARx memory (It is BAR0 here) 
 *	- lock the memory for the device
 * \return Returns 0 on success and Error code on failure.
 */ 
static int gmac_apb_probe(struct platform_device *pdev)
{
	struct resource *res;
  
	/* Do probing type stuff here.  
	 * Like calling request_region();
	 */ 
	TR2("ENABLING DEVICE\n");
	synop_apb_using_dac = 0;
	
	TR2("32 bit double-address cycle Supported with this device\n");
	
    /* Get the resource start address*/ 
    res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		printk("need a memory resource\n");
		return -ENXIO;
	}
	
	synopGMACMappedAddrSize = (u32)(res->end - res->start);

    
    if ( (request_mem_region ((u32) res->start , synopGMACMappedAddrSize,"synop_gmac_memory")) < 0) {

		synopGMACMappedAddr = 0;	// Errored in checking memory region   
		printk("Unable to request memory region for address = 0x%x size = 0x%x!!\n", res->start  , synopGMACMappedAddrSize);
		return -EBUSY;
	}

    synopGMACMappedAddr = (u8 *)  ioremap_nocache ((u32) res->start , (size_t) (synopGMACMappedAddrSize));
    
  
	if (!synopGMACMappedAddr){

		release_mem_region ((u32) res->start , synopGMACMappedAddrSize); 
		TR0("ioremap_nocache failed with addrrss %08x\n", (u32) synopGMACMappedAddr);
		return -EBUSY;
	}
	
	TR2("Physical address = %08x\n", res->start);
	TR2("Remapped address = %08x\n", (u32) synopGMACMappedAddr);

    /*Now apb interface is ready. Let give this information the the HOST module */ 
    synopGMACapbdev = pdev;
	return 0;			// Everything is fine. So return 0.
}


/**
 * remove function of Linux apb driver.
 * 
 * 	- Releases the memory allocated by probe function
 *	- Unmaps the memory region 
 *
 * \return Returns 0 on success and Error code on failure.
 */ 
static int __exit gmac_apb_remove(struct platform_device *pdev)
{
	/* Do the reverse of what probe does */ 
	if (synopGMACMappedAddr)
	{
	  TR2("Releaseing synopGMACMappedAddr 0x%x whose size is %d\n", (u32) synopGMACMappedAddr, synopGMACMappedAddrSize);
	  
	  /*release the memory region which we locked using request_mem_region */ 
	  release_mem_region ((u32) synopGMACMappedAddr, synopGMACMappedAddrSize);
	}
	TR2("Unmapping synopGMACMappedAddr =0x%x\n", (u32) synopGMACMappedAddr);
	iounmap (synopGMACMappedAddr);

	return 0;
}

static int gmac_suspend(struct device *dev)
{
	netif_device_detach(synopGMACadapter->synopGMACnetdev);
	clk_disable(synopGMACadapter->clk);

	return 0;
}

static int gmac_resume(struct device *dev)
{
	clk_enable(synopGMACadapter->clk);
	netif_device_attach(synopGMACadapter->synopGMACnetdev);

	return 0;
}

static struct dev_pm_ops gmac_pm_ops = {
	.suspend = gmac_suspend,
	.resume = gmac_resume,
};

static struct platform_driver apb_driver = {
	.remove = __exit_p(gmac_apb_remove),
	.probe = gmac_apb_probe,
	.driver = {
		.name = synopGMAC_driver_name,
		.owner = THIS_MODULE,
		.pm = &gmac_pm_ops,
	},
};

/**
 * Function to initialize the Linux apb Bus Interface.
 * Registers the platform_driver 
 * \return Returns 0 on success and Error code on failure.
 */ 

s32 __init synopGMAC_init_apb_bus_interface(void)
{
	s32 retval;
	TR2("Now Going to Call platform_register_driver\n");
	if (synopGMACMappedAddr)
	return -EBUSY;
	if ((retval = platform_driver_register(&apb_driver)))
	{
	  return retval;
	}
	if (!synopGMACMappedAddr)
	{
	  platform_driver_unregister (&apb_driver);
	  return -ENODEV;
	}
	return 0;

	
}

/**
 * Function to De-initialize the Linux apb Bus Interface.
 * 
 * Unregisters the platform_driver 
 *
 * \return Returns 0 on success and Error code on failure.
 */ 
void __exit synopGMAC_exit_apb_bus_interface(void)
{
	TR0 ("Now Calling apb_unregister_driver\n");
	platform_driver_unregister(&apb_driver);
}


