/*
 *  linux/arch/arm/mach-dmw/ccu.c
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/ccu.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <mach/platform.h>

/*
 * Adjust this to the memory controller FIFO depth (in words) to get best
 * performance.
 */
#define FIFO_DEPTH	16

#define CCU_CONT               0x000
#define CCU_CONT_MODE_RaW      (0ul << 17)
#define CCU_CONT_MODE_WriteN   (1ul << 17)
#define CCU_CONT_SWEN          (1ul << 16)
#define CCU_STAT               0x004
#define CCU_STAT_STATE_MASK    (3ul << 30)
#define CCU_STAT_STATE_IDLE    (0ul << 30)
#define CCU_STAT_STATE_WAIT    (1ul << 30)
#define CCU_STAT_STATE_BUSY    (2ul << 30)
#define CCU_STAT_ERROR         (1ul << 29)
#define CCU_STAT_SW_REQ        (1ul << 16)
#define CCU_REQ                0x008
#define CCU_SWTRIG             0x00C
#define CCU_SWTRIG_MAGIC       0x90EE
#define CCU_DUMADR             0x010
#define CCU_DUMDAT             0x014
#define CCU_RDAT               0x018

#define CCU_REG_SIZE           0x020

static void *ccu_base;
static void *ccu_victim;

/*
 * Initiate a CCU transaction to guarantee coherency wrt. writes of other bus
 * masters. Might be called recursively from different threads.
 */
void ccu_barrier(void)
{
	/* wait for CCU to be idle */
	while (readl(ccu_base + CCU_STAT) & CCU_STAT_SW_REQ)
		cpu_relax();

	/* initiate coherency transaction */
	writel(CCU_SWTRIG_MAGIC, ccu_base + CCU_SWTRIG);

	/* wait for CCU to finish request */
	while (readl(ccu_base + CCU_STAT) & CCU_STAT_SW_REQ)
		cpu_relax();
}
EXPORT_SYMBOL(ccu_barrier);

static int __init ccu_init(void)
{
	if (!request_mem_region(DMW_CCU0_BASE, CCU_REG_SIZE, "ccu0"))
		panic("CCU0: Failed to request memory region!\n");

	ccu_base = ioremap_nocache(DMW_CCU0_BASE, CCU_REG_SIZE);
	if (!ccu_base)
		panic("CCU0: Failed to map registers!\n");

	ccu_victim = kmalloc(FIFO_DEPTH*4, GFP_KERNEL);
	if (!ccu_victim)
		panic("CCU0: OOM!\n");

	/* Just enable software requests with 'N Writes' mode */
	writel(((FIFO_DEPTH-1) << 18) | CCU_CONT_MODE_WriteN | CCU_CONT_SWEN,
		ccu_base + CCU_CONT);
	writel(virt_to_phys(ccu_victim), ccu_base + CCU_DUMADR);
	writel(0xDEADBEEF, ccu_base + CCU_DUMDAT);

	return 0;
}

arch_initcall(ccu_init);
