/*
 *  linux/arch/arm/mach-dmw/arch.c
 *
 *  Copyright (C) 2010-2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/init.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/platform_device.h>
#include <linux/sysdev.h>
#include <linux/interrupt.h>
#include <linux/amba/bus.h>
#include <linux/amba/clcd.h>
#include <linux/delay.h>
#include <linux/ctype.h>
#include <linux/memblock.h>
#include <linux/regulator/machine.h>

#include <asm/system.h>
#include <mach/hardware.h>
#include <mach/clock.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/hardware/arm_timer.h>
#include <asm/mach-types.h>
#include <asm/gpio.h>
#include <asm/setup.h>
#include <asm/memory.h>

#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/time.h>
#include <asm/mach/map.h>

#include "arch.h"
#include "irq.h"
#include "devices.h"

static char *dmw_board = "unknown";
static char *dmw_board_features = "";
static bool dmw_nodebug = false;

static int __init dmw_detect_board_commandline(char *line)
{
	dmw_board = line;
	line = strchr(dmw_board, ':');
	if (line) {
		*line++ = '\0';
		dmw_board_features = line;
	}

	return 1;
}

__setup("board=", dmw_detect_board_commandline);

char *dmw_board_get_type(void)
{
	return dmw_board;
}
EXPORT_SYMBOL(dmw_board_get_type);

bool dmw_board_is(char *board)
{
	return strcmp(dmw_board, board) == 0;
}
EXPORT_SYMBOL(dmw_board_is);

bool dmw_board_has_feature(char *feature)
{
	int len;
	char *tmp_board_features;
	char *tmp_feature_name;
	char *spot;

	len = strlen(dmw_board_features) + 3;

	tmp_board_features = (char *)kmalloc(len, GFP_KERNEL);
	if (!tmp_board_features)
		return false;

	tmp_feature_name = (char *)kmalloc(len, GFP_KERNEL);
	if (!tmp_feature_name) {
		kfree(tmp_board_features);
		return false;
	}

	sprintf(tmp_board_features, ",%s,", dmw_board_features);
	sprintf(tmp_feature_name, ",%s,", feature);

	spot = strstr(tmp_board_features, tmp_feature_name);

	kfree(tmp_board_features);
	kfree(tmp_feature_name);

	if (!spot)
		return false;

	return true;
}

EXPORT_SYMBOL(dmw_board_has_feature);

static int dmw_chip;
static int dmw_chip_rev;

static int dmw_detect_chip_rev_96(void)
{
	int ret;
	void *rom;

	ret = readl(IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_CHIP_REV) & 0xffff;
	if (ret > 0)
		return ret + 1;

	/*
	 * Revision A and B share the same code in SYSCFG CHIP_REV. :( Look
	 * into the bootrom code to distinguis both.
	 */
	rom = ioremap_nocache(DMW_BOOTROM_BASE, SZ_128K);
	if (!rom)
		return -1;

	switch (readb(rom + 68)) {
	case 0x30: ret = DMW_CHIP_DMW96_REV_A; break;
	case 0x90: ret = DMW_CHIP_DMW96_REV_B; break;
	default:   ret = -1;
	}

	iounmap(rom);
	return ret;
}

static void dmw_detect_chip(void)
{
	unsigned id;

	if (dmw_chip)
		return;

	id = readl(IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_CHIP_ID);

	switch (id) {
	case 0x0960:
		dmw_chip = DMW_CHIP_DMW96;
		dmw_chip_rev = dmw_detect_chip_rev_96();
		break;
	default:
		dmw_chip = DMW_CHIP_UNKNOWN;
	}
}

int dmw_get_chip(void)
{
	dmw_detect_chip();
	return dmw_chip;
}
EXPORT_SYMBOL(dmw_get_chip);

int dmw_get_chip_rev(void)
{
	dmw_detect_chip();
	return dmw_chip_rev;
}
EXPORT_SYMBOL(dmw_get_chip_rev);


bool dmw_debug_disabled(void)
{
	return dmw_nodebug;
}

static int __init nodebug_setup(char *__unused)
{
	unsigned long reg;

	dmw_nodebug = true;

	/*
	 * Disable SYSCFG CoreSight features...
	 */
	reg = readl(IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_GCR1);
	reg &= ~(1 << 11); /* DBGNOCLOCKSTOP */
	writel(reg, IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_GCR1);

	reg = readl(IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_POWER_EN);
	reg &= ~(1 << 17); /* DBG_CLK_ON_OVR */
	reg &= ~(1 << 16); /* TRACE_BUF_ON_OVR */
	writel(reg, IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_POWER_EN);

	return 1;
}

__setup("nodebug", nodebug_setup);

/*
 * All IO addresses are mapped onto VA 0xFFFx.xxxx, where x.xxxx
 * is the (PA >> 12).
 */
static struct map_desc dmw_io_desc[] __initdata = {
	{
		.virtual	= IO_ADDRESS(DMW_PLICU_BASE),
		.pfn		= __phys_to_pfn(DMW_PLICU_BASE),
		.length		= SZ_4K,
		.type		= MT_DEVICE
	}, {
		.virtual	= IO_ADDRESS(DMW_CMU_BASE),
		.pfn		= __phys_to_pfn(DMW_CMU_BASE),
		.length		= SZ_4K,
		.type		= MT_DEVICE
	}, {
		.virtual	= IO_ADDRESS(DMW_SYSCFG_BASE),
		.pfn		= __phys_to_pfn(DMW_SYSCFG_BASE),
		.length		= SZ_4K,
		.type		= MT_DEVICE
	},
};

static void __init dmw_map_io(void)
{
	iotable_init(dmw_io_desc, ARRAY_SIZE(dmw_io_desc));
}

static void __init dmw_early_init(void)
{
	if (dmw_clk_init())
		panic("Clock init failed!");
}

static void __init dmw_init(void)
{
	WARN(strcmp(dmw_board, "unknown") == 0, "No valid board type specified");

	printk("Board: %s\n", dmw_board);
	printk("Board features: %s\n", dmw_board_features);

	regulator_has_full_constraints();

	/*
	 * build-time check for correct CONSISTENT_DMA_SIZE; the SZ_4M is for
	 * the peripherals in dmw_io_desc[] (IO_ADDRESS() mappings start at
	 * VMALLOC_END).
	 */
	BUILD_BUG_ON(VMALLOC_END + (SZ_4M) > CONSISTENT_BASE);
}

extern struct sys_timer dmw_timer;

static phys_addr_t dmw_reservemem_base;
static phys_addr_t dmw_reservemem_next;
static size_t dmw_reservemem_size;

static size_t __init dmw_reservemem_remain(void)
{
	size_t used = dmw_reservemem_next - dmw_reservemem_base;
	return dmw_reservemem_size - used;
}

int __init dmw_reservemem_get(phys_addr_t *pa, size_t size)
{
	size = roundup(size, SZ_1M);

	if (dmw_reservemem_remain() < size)
		return -ENOMEM;

	*pa = dmw_reservemem_next;
	dmw_reservemem_next += size;
	return 0;
}

static void __init dmw_reserve(void)
{
	dmw_reservemem_size =
		DMW_RESERVEMEM_GPU +
		DMW_RESERVEMEM_CAMERA +
		DMW_RESERVEMEM_MEMALLOC +
		DMW_RESERVEMEM_CSS;

	dmw_reservemem_base = dmw_reservemem_next =
		memblock_alloc(dmw_reservemem_size, SZ_1M);
	memblock_free(dmw_reservemem_base, dmw_reservemem_size);
	memblock_remove(dmw_reservemem_base, dmw_reservemem_size);
}

MACHINE_START(DMW96, "DSPG DMW96")
	.boot_params	= 0x40000100,
	.map_io		= dmw_map_io,
	.init_early	= dmw_early_init,
	.init_machine	= dmw_init,
	.init_irq	= dmw_init_irq,
	.timer		= &dmw_timer,
	.reserve	= dmw_reserve,
MACHINE_END

